package CSHL::ComparativeMaps::Admin;

#-----------------------------------------------------
# $Id: Admin.pm,v 1.11 2002/03/27 20:16:21 kclark Exp $
#
# File       : Admin.pm
# Programmer : Ken Y. Clark, kclark@logsoft.com
# Created    : 2001/12/17
# Purpose    : curate comparative map data
#-----------------------------------------------------

use strict;
use Data::Dumper;

use Apache::Constants qw[ :common M_GET REDIRECT ];
use Apache::Request;
use Apache::SubProcess;
use POSIX 'setsid';
use Template;

use CSHL::Config;
use CSHL::ComparativeMaps::Constants;

use constant ADMIN_HOME_URI  => '/maps/admin';

use constant TEMPLATE        => {
    admin_home               => 'admin_home.tmpl',
    error                    => 'admin_error.tmpl',
    confirm_delete           => 'admin_confirm_delete.tmpl',
    import_instructions      => 'admin_import_instructions.tmpl',
    feature_types_view       => 'admin_feature_types_view.tmpl',
    feature_type_create      => 'admin_feature_type_create.tmpl',
    feature_type_edit        => 'admin_feature_type_edit.tmpl',
    genetic_map_view         => 'admin_genetic_map_view.tmpl',
    map_studies_view         => 'admin_map_studies_view.tmpl',
    map_study_create         => 'admin_map_study_create.tmpl',
    map_study_edit           => 'admin_map_study_edit.tmpl',
    map_study_import_report  => 'admin_map_study_import_report.tmpl',
    map_study_view           => 'admin_map_study_view.tmpl',
    map_type_edit            => 'admin_map_type_edit.tmpl',
    map_type_create          => 'admin_map_type_create.tmpl',
    map_types_view           => 'admin_map_types_view.tmpl',
    parent_role_create       => 'admin_parent_role_create.tmpl',
    parent_role_edit         => 'admin_parent_role_edit.tmpl',
    parent_roles_view        => 'admin_parent_roles_view.tmpl',
    physical_map_view        => 'admin_physical_map_view.tmpl',
    species_edit             => 'admin_species_edit.tmpl',
    species_create           => 'admin_species_create.tmpl',
    species_view             => 'admin_species_view.tmpl',
    update_physical_features => 'admin_update_physical_features.tmpl',  
};

use vars qw( $VERSION %DISPATCH );

$VERSION = (qw$Revision: 1.11 $)[-1];

%DISPATCH = (
    admin_home               => \&admin_home,
    confirm_delete           => \&confirm_delete,
    default                  => \&admin_home,
    entity_delete            => \&entity_delete,
    import_instructions      => \&import_instructions,
    genetic_map_view         => \&genetic_map_view,
    feature_types_view       => \&feature_types_view,
    feature_type_create      => \&feature_type_create,
    feature_type_insert      => \&feature_type_insert,
    feature_type_edit        => \&feature_type_edit,
    feature_type_update      => \&feature_type_update,
    map_studies_view         => \&map_studies_view,
    map_study_create         => \&map_study_create,
    map_study_edit           => \&map_study_edit,
    map_study_insert         => \&map_study_insert,
    map_study_import         => \&map_study_import,
    map_study_update         => \&map_study_update,
    map_study_view           => \&map_study_view,
    map_type_create          => \&map_type_create,
    map_type_edit            => \&map_type_edit,
    map_type_insert          => \&map_type_insert,
    map_type_update          => \&map_type_update,
    map_types_view           => \&map_types_view,
    parent_role_create       => \&parent_role_create,
    parent_role_insert       => \&parent_role_insert,
    parent_role_edit         => \&parent_role_edit,
    parent_role_update       => \&parent_role_update,
    parent_roles_view        => \&parent_roles_view,
    physical_map_view        => \&physical_map_view,
    species_create           => \&species_create,
    species_edit             => \&species_edit,
    species_insert           => \&species_insert,
    species_update           => \&species_update,
    species_view             => \&species_view,
    update_physical_features => \&update_physical_features,
);

#-----------------------------------------------------
sub handler {
    #
    # Make a jazz noise here...
    #
    my $r            = shift;
    my $apr          = Apache::Request->new( $r->is_main ? $r : $r->main );
    my $template_dir = $apr->dir_config('TEMPLATE_DIR') || TEMPLATE_DIR;
    my $t            = Template->new( { INCLUDE_PATH => $template_dir } );
    my $db           = DBI->connect( 
        MapDataSource,
        MapDBUser,
        MapDBPassword,
        MapDBOptions,
    );
    my $return;

    my $flush_val = $|;

    eval {
        my $action = $apr->param( 'action' ) || '';
        $action    = 'default' unless defined $DISPATCH{ $action };
        $return    = $DISPATCH{ $action }->( $apr, $db, $t );
    };

    if ( my $error = $@ ) { 
        my $html;
        $t->process( TEMPLATE->{'error'}, { msg => $error }, \$html )
            or $html = $t->error;
        $apr->content_type('text/html');
        $apr->send_http_header;
        $apr->print( $html );
        $return = OK;
    }

    return $return || OK;
}

#-----------------------------------------------------
sub _extract_numbers {
#
# Returns just the numbers in a string
#
    my $arg = shift;
    $arg =~ s/[^\d]//g;
    return $arg;
}


#-----------------------------------------------------
sub admin_home {
    my ( $apr, $db, $t ) = @_;
    my $order_by = $apr->param( 'order_by' ) || 'species_name';
    unless ( $order_by eq 'map_study_name' ) {
        $order_by .= ',map_study_name';
    }

    my $map_studies = $db->selectall_arrayref(
        qq[
            select   ms.map_study_id, ms.map_study_name,
                     s.common_name as species_name,
                     mt.map_type
            from     cmap_map_study ms, cmap_species s, cmap_map_type mt
            where    ms.species_id=s.species_id
            and      ms.map_type_id=mt.map_type_id
            order by $order_by
        ], { Columns => {} }
    );
    my $output; 
    $t->process( 
        TEMPLATE->{'admin_home'}, 
        { map_studies => $map_studies }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub confirm_delete {
    my ( $apr, $db, $t )    = @_;
    my $entity_type       = $apr->param('entity_type') or die 'No entity type';
    my $entity_id         = $apr->param('entity_id')   or die 'No entity id';
    my $entity_name       = $apr->param('entity_name') || '';

    unless ( $entity_name ) {
        (my $base_table_name   = $entity_type ) =~ s/^cmap_//;
        my $entity_id_field   = $apr->param('entity_id_field') || 
                                $base_table_name.'_id';
        my $entity_name_field = $apr->param('entity_name_field') || 
                                $base_table_name.'_name';
        $entity_name          = $db->selectrow_array(
            qq[
                select $entity_name_field
                from   $entity_type
                where  $entity_id_field=$entity_id
            ]
        );
    }

    my $entity = {
        id     => $entity_id,
        name   => $entity_name,
        type   => $entity_type, 
    };

    my $output; 
    $t->process( TEMPLATE->{'confirm_delete'}, { entity => $entity }, \$output )
        or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub entity_delete {
    my ( $apr, $db, $t ) = @_;
    my $entity_type      = $apr->param('entity_type') or die 'No entity type';
    my $entity_id        = $apr->param('entity_id')   or die 'No entity id';
    my $uri_args;

    #
    # Map Study
    #
    if ( $entity_type eq 'cmap_map_study' ) {
        my $map_study_id = $entity_id;

        my $map_type = uc $db->selectrow_array(
            q[
                select mt.map_type
                from   cmap_map_study ms, 
                       cmap_map_type mt
                where  ms.map_study_id=?
                and    ms.map_type_id=mt.map_type_id
            ]
        );

        my $sql = $map_type eq GENETIC 
            ? q[
                select genetic_map_id 
                from   cmap_genetic_map
                where  map_study_id=?
            ]
            : q[
                select physical_map_id 
                from   cmap_physical_map
                where  map_study_id=?
            ]
        ;

        my @map_ids = @{
            $db->selectcol_arrayref( $sql, {}, ( $map_study_id ) )
        };

        if ( @map_ids ) {
            my ( $table, $field) = $map_type eq GENETIC 
                ? qw[ cmap_genetic_map_position  genetic_map_id  ]
                : qw[ cmap_physical_map_position physical_map_id ]
            ;

            $db->do(
                qq[
                    delete 
                    from   $table
                    where  $field in (].
                           join( ',', @map_ids ).q[)
                ]
            );
        }

        for my $table ( 
            qw[ cmap_genetic_map cmap_physical_map cmap_map_study ] 
        ) {
            $db->do(
                qq[
                    delete 
                    from   $table
                    where  map_study_id=?
                ],
                {}, ( $map_study_id )
            );
        }
    }
    #
    # Map Type
    #
    elsif ( $entity_type eq 'cmap_map_type' ) {
        my $map_type_id = $apr->param('entity_id') or die 'No map type id';

        my $sth = $db->prepare(
            q[
                select   count(ms.map_study_id) as count, mt.map_type
                from     cmap_map_study ms, cmap_map_type mt
                where    ms.map_type_id=?
                and      ms.map_type_id=mt.map_type_id
                group by map_type
            ]
        );
        $sth->execute( $map_type_id );
        my $hr = $sth->fetchrow_hashref;

        if ( $hr->{'count'} > 0 ) {
            die "Unable to delete map type &quot;", $hr->{'map_type'}, 
                "&quot; as ", $hr->{'count'}, 
                " map studies are linked to it.\n";
        }
        else {
            $db->do(
                q[
                    delete
                    from   cmap_map_type
                    where  map_type_id=?
                ], 
                {}, ( $map_type_id ) 
            );
            $uri_args = '?action=map_types_view';
        }
    }
    #
    # Species
    #
    elsif ( $entity_type eq 'cmap_species' ) {
        my $species_id = $apr->param('entity_id') or die 'No map type id';

        my $sth = $db->prepare(
            q[
                select   count(ms.map_study_id) as count, s.common_name
                from     cmap_map_study ms, cmap_species s
                where    s.species_id=?
                and      ms.species_id=s.species_id
                group by common_name
            ]
        );
        $sth->execute( $species_id );
        my $hr = $sth->fetchrow_hashref;

        if ( $hr->{'count'} > 0 ) {
            die "Unable to delete species &quot;", $hr->{'common_name'}, 
                "&quot; as ", $hr->{'count'}, 
                " map studies are linked to it.\n";
        }
        else {
            $db->do(
                q[
                    delete
                    from   cmap_species
                    where  species_id=?
                ], 
                {}, ( $species_id ) 
            );
            $uri_args = '?action=species_view';
        }
    }
    #
    # Parent Role
    #
    elsif ( $entity_type eq 'cmap_parent_role' ) {
        my $parent_role_id = $apr->param('entity_id') 
            or die 'No parent role id';

        my $sth = $db->prepare(
            q[
                select   count(ms.parent_1_role_id) as count, pr.role
                from     cmap_map_study ms, 
                         cmap_parent_role pr
                where    ms.parent_1_role_id=?
                and      ms.parent_1_role_id=pr.parent_role_id
                group by role
            ]
        );
        $sth->execute( $parent_role_id );
        my $hr = $sth->fetchrow_hashref;

        if ( $hr->{'count'} > 0 ) {
            die "Unable to delete parent role &quot;", $hr->{'role'}, 
                "&quot; as ", $hr->{'count'}, 
                " map studies are linked to it.\n";
        }
        else {
            $db->do(
                q[
                    delete
                    from   cmap_parent_role
                    where  parent_role_id=?
                ], 
                {}, ( $parent_role_id ) 
            );
            $uri_args = '?action=parent_roles_view';
        }
    }
    #
    # Feature Type
    #
    elsif ( $entity_type eq 'cmap_feature_type' ) {
        my $feature_type_id = $apr->param('entity_id') 
            or die 'No feature type id';

        my $sth = $db->prepare(
            q[
                select   count(f.feature_type_id) as count, ft.feature_type
                from     cmap_feature f, 
                         cmap_feature_type ft
                where    f.feature_type_id=?
                and      f.feature_type_id=ft.feature_type_id
                group by feature_type
            ]
        );
        $sth->execute( $feature_type_id );
        my $hr = $sth->fetchrow_hashref;

        if ( $hr->{'count'} > 0 ) {
            die "Unable to delete feature type &quot;", $hr->{'feature_type'}, 
                "&quot; as ", $hr->{'count'}, 
                " features are linked to it.\n";
        }
        else {
            $db->do(
                q[
                    delete
                    from   cmap_feature_type
                    where  feature_type_id=?
                ], 
                {}, ( $feature_type_id ) 
            );
            $uri_args = '?action=feature_types_view';
        }
    }
    else {
        die "You are not allowed to delete entities of type $entity_type.";
    }

    return redirect_home( $apr, ADMIN_HOME_URI.$uri_args ); 
    return OK;
}

#-----------------------------------------------------
sub genetic_map_view {
    my ( $apr, $db, $t ) = @_;
    my $genetic_map_id   = $apr->param('genetic_map_id');
    my $order_by         = $apr->param('order_by') || 'position_start';

    my $sth = $db->prepare(
        q[
            select map.genetic_map_id, map.linkage_group,
                   ms.map_study_id, ms.map_study_name,
                   mt.map_type
            from   cmap_genetic_map map, cmap_map_study ms, cmap_map_type mt
            where  map.genetic_map_id=?
            and    map.map_study_id=ms.map_study_id
            and    ms.map_type_id=mt.map_type_id
        ]
    );
    $sth->execute( $genetic_map_id );
    my $map = $sth->fetchrow_hashref;

    $map->{'features'} = $db->selectall_arrayref(
        qq[
            select   mp.genetic_map_position_id, mp.map_position_name,
                     mp.genetic_map_id, mp.position_start, mp.position_stop,
                     f.feature_id, f.feature_name, f.feature_type_id,
                     ft.feature_type
            from     cmap_genetic_map_position mp, cmap_feature f, 
                     cmap_feature_type ft
            where    mp.genetic_map_id=?
            and      mp.feature_id=f.feature_id
            and      f.feature_type_id=ft.feature_type_id
            order by $order_by
        ],
        { Columns => {} }, ( $genetic_map_id )
    );

    my $output; 
    $t->process( TEMPLATE->{'genetic_map_view'}, { map => $map }, \$output ) 
        or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub feature_type_create {
    my ( $apr, $db, $t ) = @_;

    my $output; 
    $t->process( 
        TEMPLATE->{'feature_type_create'},
        { 
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub feature_type_edit {
    my ( $apr, $db, $t ) = @_;

    my $sth = $db->prepare(
        q[
            select   feature_type_id, feature_type
            from     cmap_feature_type
            where    feature_type_id=?
        ]
    );
    $sth->execute( $apr->param('feature_type_id') );
    my $feature_type = $sth->fetchrow_hashref;

    my $output; 
    $t->process( 
        TEMPLATE->{'feature_type_edit'},
        { 
            feature_type => $feature_type,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}


#-----------------------------------------------------
sub feature_type_insert {
    my ( $apr, $db, $t ) = @_;
    my $feature_type     = $apr->param('feature_type') or die 'No feature type';
    my $feature_type_id = next_number(
        db          => $db, 
        table_name  => 'cmap_feature_type',
        id_field    => 'feature_type_id',
    ) or die 'No feature type id';

    $db->do(
        q[ 
            insert
            into   cmap_feature_type ( feature_type_id, feature_type )
            values ( ?, ? )
        ],
        {}, 
        ( $feature_type_id, $feature_type )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=feature_types_view' ); 
}

#-----------------------------------------------------
sub feature_type_update {
    my ( $apr, $db, $t ) = @_;
    my $feature_type_id  = $apr->param('feature_type_id') 
        or die 'No feature type id';
    my $feature_type     = $apr->param('feature_type')    
        or die 'No feature type';

    $db->do(
        q[ 
            update cmap_feature_type
            set    feature_type=?
            where  feature_type_id=?
        ],
        {}, ( $feature_type, $feature_type_id )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=feature_types_view' ); 
}

#-----------------------------------------------------
sub feature_types_view {
    my ( $apr, $db, $t ) = @_;
    my $feature_types = $db->selectall_arrayref(
        q[
            select   f.feature_type_id, f.feature_type
            from     cmap_feature_type f
            order by feature_type
        ], 
        { Columns => {} }
    );
    my $output; 
    $t->process( 
        TEMPLATE->{'feature_types_view'}, 
        { feature_types => $feature_types }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub import_instructions {
    my ( $apr, $db, $t ) = @_;
    my $map_study_id = $apr->param('map_study_id') or die 'No map study id';
    my $map_study_name = $db->selectrow_array(
        q[
            select map_study_name
            from   cmap_map_study
            where  map_study_id=?
        ],
        {}, ( $map_study_id )
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'import_instructions'}, 
        { 
            map_study_id   => $map_study_id,
            map_study_name => $map_study_name,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_studies_view {
    my ( $apr, $db, $t ) = @_;
    my $map_type_id = $apr->param( 'map_type_id' ) ||             '';
    my $species_id  = $apr->param( 'species_id'  ) ||             '';
    my $order_by    = $apr->param( 'order_by' )    || 'species_name';

    unless ( $order_by eq 'map_study_name' ) {
        $order_by .= ',map_study_name';
    }

    my $sql = q[
        select   ms.map_study_id, ms.map_study_name,
                 s.common_name as species_name,
                 mt.map_type
        from     cmap_map_study ms, 
                 cmap_species s, 
                 cmap_map_type mt
        where    ms.species_id=s.species_id
        and      ms.map_type_id=mt.map_type_id
    ];
    $sql .= qq[ and ms.map_type_id=$map_type_id ] if $map_type_id;
    $sql .= qq[ and ms.species_id=$species_id ]   if $species_id ;
    $sql .= qq[ order by $order_by ];
    my $map_studies = $db->selectall_arrayref( $sql, { Columns => {} } );

    my $specie = $db->selectall_arrayref(
        q[
            select   distinct s.species_id, s.full_name, s.common_name
            from     cmap_species s,
                     cmap_map_study ms
            where    s.species_id=ms.species_id
            order by common_name
        ], { Columns => {} }
    );

    my $map_types = $db->selectall_arrayref(
        q[
            select   distinct mt.map_type_id, mt.map_type
            from     cmap_map_type mt,
                     cmap_map_study ms
            where    mt.map_type_id=ms.map_type_id
            order by map_type
        ], { Columns => {} }
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'map_studies_view'}, 
        { 
            apr         => $apr,
            specie      => $specie,
            map_types   => $map_types,
            map_studies => $map_studies,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_study_create {
    my ( $apr, $db, $t, $errors ) = @_;

    my $specie = $db->selectall_arrayref(
        q[
            select   s.species_id, s.full_name, s.common_name
            from     cmap_species s
            order by common_name
        ], { Columns => {} }
    );

    my $map_types = $db->selectall_arrayref(
        q[
            select   mt.map_type_id, mt.map_type
            from     cmap_map_type mt
            order by map_type
        ], { Columns => {} }
    );

    my $parent_roles = $db->selectall_arrayref(
        q[
            select   parent_role_id, role
            from     cmap_parent_role
            order by role
        ], { Columns => {} }
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'map_study_create'},
        { 
            apr          => $apr,
            errors       => $errors,
            specie       => $specie,
            map_types    => $map_types,
            parent_roles => $parent_roles,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_study_edit {
    my ( $apr, $db, $t ) = @_;

    my $sth = $db->prepare(
        q[
            select    ms.map_study_id, ms.map_study_name,
                      ms.short_name, ms.remarks,
                      ms.map_type_id, ms.species_id,
                      ms.parent_1, ms.parent_2, 
                      ms.population_size, ms.parent_1_role_id,
                      s.common_name as species_common_name,
                      s.full_name as species_full_name,
                      mt.map_type, mt.map_units,
                      pr.role
            from      cmap_map_study ms, 
                      cmap_species s, 
                      cmap_map_type mt
            left join cmap_parent_role pr 
            on        ms.parent_1_role_id=pr.parent_role_id
            where     ms.species_id=s.species_id
            and       ms.map_type_id=mt.map_type_id
            and       ms.map_study_id=?
        ],
    );

    $sth->execute( $apr->param( 'map_study_id' ) );
    my $map_study = $sth->fetchrow_hashref;

    my $specie = $db->selectall_arrayref(
        q[
            select   species_id, full_name, common_name
            from     cmap_species
            order by common_name
        ], { Columns => {} }
    );

    my $map_types = $db->selectall_arrayref(
        q[
            select   map_type_id, map_type, map_units
            from     cmap_map_type
            order by map_type
        ], { Columns => {} }
    );

    my $parent_roles = $db->selectall_arrayref(
        q[
            select   parent_role_id, role
            from     cmap_parent_role
            order by role
        ], { Columns => {} }
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'map_study_edit'},
        { 
            map_study    => $map_study,
            specie       => $specie,
            map_types    => $map_types,
            parent_roles => $parent_roles,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_study_insert {
    my ( $apr, $db, $t ) = @_;
    my @errors           = ();
    my $map_study_name   = $apr->param('map_study_name')
        or push @errors, 'No map study name';
    my $short_name       = $apr->param('short_name')
        or push @errors, 'No short name';
    my $species_id       = $apr->param('species_id')
        or push @errors, 'No species';
    my $map_type_id      = $apr->param('map_type_id')
        or push @errors, 'No map type';
    my $parent_1_role_id = $apr->param('parent_1_role_id');
    my $parent_1         = $apr->param('parent_1');
    my $parent_2         = $apr->param('parent_2');
    my $population_size  = $apr->param('population_size');
    my $remarks          = $apr->param('remarks');

    if ( @errors ) {
        return map_study_create( $apr, $db, $t, \@errors );
    }

    my $map_study_id = next_number(
        db           => $db, 
        table_name   => 'cmap_map_study',
        id_field     => 'map_study_id',
    ) or die 'No map study id';

    $db->do(
        q[
            insert
            into   cmap_map_study
                   ( map_study_id, map_study_name, short_name,
                     species_id, map_type_id, parent_1_role_id,
                     parent_1, parent_2, population_size, remarks 
                   )
            values ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ? )
        ],
        {}, 
        ( 
            $map_study_id, $map_study_name, $short_name,
            $species_id, $map_type_id, $parent_1_role_id,
            $parent_1, $parent_2, $population_size, $remarks 
        )
    );

    return redirect_home( 
        $apr, 
        ADMIN_HOME_URI."?action=map_study_view;map_study_id=$map_study_id",
    );
}

#-----------------------------------------------------
sub map_study_import {
    my ( $apr, $db, $t ) = @_;

    my $map_study_id = $apr->param('map_study_id');
    my $upload       = $apr->upload or die 'No import file';
    my $fh           = $upload->fh;
    my $importer     = CSHL::ComparativeMaps::Admin::DataImport->new(db=>$db);
    my $ok           = $importer->import( 
        map_study_id => $map_study_id,
        fh           => $fh,
        apr          => $apr,
    );

#    my $output; 
#    $t->process( 
#        TEMPLATE->{'map_study_import_report'},
#        { 
#            import_ok    => $ok,
#            report       => [ $importer->report ],
#            map_study_id => $map_study_id,
#        }, 
#        \$output 
#    ) or $output = $t->error;

#    $apr->content_type('text/html');
#    $apr->send_http_header;
#    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_study_view {
    my ( $apr, $db, $t ) = @_;
    my $map_study_id     = $apr->param('map_study_id') or die 'No map study id';

    my $sth = $db->prepare(
        q[
            select    ms.map_study_id, ms.map_study_name,
                      ms.short_name, ms.remarks,
                      ms.map_type_id, ms.species_id, ms.citation_id, 
                      ms.parent_1, ms.parent_2,
                      ms.parent_1_role_id,
                      ms.population_size,
                      s.common_name as species_common_name,
                      s.full_name as species_full_name,
                      mt.map_type, mt.map_units,
                      pr.role as parent_1_role
            from      cmap_map_study ms, 
                      cmap_species s, 
                      cmap_map_type mt
            left join cmap_parent_role pr 
            on        ms.parent_1_role_id=pr.parent_role_id
            where     ms.species_id=s.species_id
            and       ms.map_type_id=mt.map_type_id
            and       ms.map_study_id=?
        ]
    );

    $sth->execute( $map_study_id );
    my $map_study = $sth->fetchrow_hashref or die 'Invalid map study id';

    my $sql;
    if ( uc $map_study->{'map_type'} eq GENETIC ) {
        $sql = q[
            select   genetic_map_id, linkage_group
            from     cmap_genetic_map
            where    map_study_id=?
            order by linkage_group
        ],
    }
    else {
        $sql = q[
            select   physical_map_id, map_name, 
                     position_start, position_stop
            from     cmap_physical_map
            where    map_study_id=?
            order by map_name
        ],
    }

    my @maps = @{ 
        $db->selectall_arrayref( $sql, { Columns => {} }, ( $map_study_id ) )
    };

    my $map_name_field = uc $map_study->{'map_type'} eq GENETIC ? 
        'linkage_group' : 'map_name';
    my $all_numbers = grep { $_->{ $map_name_field } =~ m/[0-9]/ } @maps;
    if ( $all_numbers == scalar @maps ) {
        @maps = 
            map  { $_->[0] }
            sort { $a->[1] <=> $b->[1] }
            map  { [$_, _extract_numbers( $_->{ $map_name_field } )] }
            @maps
        ;
    }

    $map_study->{'maps'} = \@maps;

    #
    # References
    #
    $map_study->{'references'} = $db->selectall_arrayref(
        q[
            select reference_id
            from   cmap_map_study_to_reference
            where  map_study_id=?
        ],
        { Columns => {} },
        ( $map_study_id )
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'map_study_view'}, { map_study => $map_study }, \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_study_update {
    my ( $apr, $db, $t ) = @_;
    my $map_study_id     = $apr->param('map_study_id');
    my $map_study_name   = $apr->param('map_study_name');
    my $short_name       = $apr->param('short_name');
    my $species_id       = $apr->param('species_id');
    my $map_type_id      = $apr->param('map_type_id');
    my $parent_1_role_id = $apr->param('parent_1_role_id');
    my $parent_1         = $apr->param('parent_1');
    my $parent_2         = $apr->param('parent_2');
    my $population_size  = $apr->param('population_size');
    my $remarks          = $apr->param('remarks');

    $db->do(
        q[
            update cmap_map_study
            set    map_study_name=?, short_name=?,
                   species_id=?, map_type_id=?,
                   parent_1_role_id=?, parent_1=?,
                   parent_2=?, population_size=?,
                   remarks=?
            where  map_study_id=?
        ],
        {},
        (
            $map_study_name, $short_name, $species_id,
            $map_type_id, $parent_1_role_id, $parent_1,
            $parent_2, $population_size, $remarks, 
            $map_study_id
        )
    );

    return redirect_home( 
        $apr, 
        ADMIN_HOME_URI."?action=map_study_view;map_study_id=$map_study_id",
    )
}

#-----------------------------------------------------
sub redirect_home {
    my ( $apr, $uri ) = @_;

    $apr->method_number( M_GET );
    $apr->method( 'GET' );
    $apr->headers_in->unset( 'Content-length' );
    $apr->headers_out->add( Location => $uri );
    $apr->status( REDIRECT );
    $apr->send_http_header;
    return DONE;
}

#-----------------------------------------------------
sub map_type_create {
    my ( $apr, $db, $t ) = @_;

    my $output; 
    $t->process( 
        TEMPLATE->{'map_type_create'},
        { 
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_type_edit {
    my ( $apr, $db, $t ) = @_;

    my $sth = $db->prepare(
        q[
            select   map_type_id, map_type, map_units
            from     cmap_map_type
            where    map_type_id=?
        ]
    );
    $sth->execute( $apr->param('map_type_id') );
    my $map_type = $sth->fetchrow_hashref;

    my $output; 
    $t->process( 
        TEMPLATE->{'map_type_edit'},
        { 
            map_type => $map_type,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub map_type_insert {
    my ( $apr, $db, $t ) = @_;
    my $map_type    = $apr->param('map_type')    or die 'No map type';
    my $map_units   = $apr->param('map_units')   or die 'No map units';
    my $map_type_id = next_number(
        db          => $db, 
        table_name  => 'cmap_map_type',
        id_field    => 'map_type_id',
    ) or die 'No map type id';

    $db->do(
        q[ 
            insert
            into   cmap_map_type ( map_type_id, map_type, map_units )
            values ( ?, ?, ? )
        ],
        {}, ( $map_type_id, $map_type, $map_units )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=map_types_view' ); 
}

#-----------------------------------------------------
sub map_type_update {
    my ( $apr, $db, $t ) = @_;
    my $map_type_id = $apr->param('map_type_id') or die 'No map type id';
    my $map_type    = $apr->param('map_type')    or die 'No map type';
    my $map_units   = $apr->param('map_units')   or die 'No map units';

    $db->do(
        q[ 
            update cmap_map_type
            set    map_type=?, map_units=?
            where  map_type_id=?
        ],
        {}, ( $map_type, $map_units, $map_type_id )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=map_types_view' ); 
}

#-----------------------------------------------------
sub map_types_view {
    my ( $apr, $db, $t ) = @_;
    my $order_by = $apr->param('order_by') || 'map_type';

    my $map_types = $db->selectall_arrayref(
        qq[
            select   map_type_id, map_type, map_units
            from     cmap_map_type
            order by $order_by
        ],
        { Columns => {} }
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'map_types_view'},
        { 
            map_types => $map_types,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub parent_role_create {
    my ( $apr, $db, $t ) = @_;

    my $output; 
    $t->process( TEMPLATE->{'parent_role_create'}, {}, \$output ) 
        or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub parent_role_insert {
    my ( $apr, $db, $t ) = @_;
    my $role             = $apr->param('role') or die 'No role';
    my $parent_role_id   = next_number(
        db          => $db, 
        table_name  => 'cmap_parent_role',
        id_field    => 'parent_role_id',
    ) or die 'No parent role id';

    $db->do(
        q[ 
            insert
            into   cmap_parent_role (parent_role_id, role)
            values ( ?, ? ) 
        ],
        {}, ( $parent_role_id, $role )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=parent_roles_view' ); 
}

#-----------------------------------------------------
sub parent_role_edit {
    my ( $apr, $db, $t ) = @_;
    my $parent_role_id   = $apr->param('parent_role_id') 
        or die 'No parent role id';

    my $sth = $db->prepare(
        q[
            select parent_role_id, role
            from   cmap_parent_role
            where  parent_role_id=?
        ]
    );
    $sth->execute( $parent_role_id );
    my $parent_role = $sth->fetchrow_hashref;

    my $output; 
    $t->process( 
        TEMPLATE->{'parent_role_edit'}, 
        { parent_role => $parent_role }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub parent_role_update {
    my ( $apr, $db, $t ) = @_;
    my $parent_role_id   = $apr->param('parent_role_id') 
        or die 'No parent role id';
    my $role             = $apr->param('role') or die 'No role';

    $db->do(
        q[ 
            update cmap_parent_role 
            set    role=?
            where  parent_role_id=?
        ],
        {}, ( $role, $parent_role_id )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=parent_roles_view' ); 
}

#-----------------------------------------------------
sub parent_roles_view {
    my ( $apr, $db, $t ) = @_;

    my $parent_roles = $db->selectall_arrayref(
        q[
            select   parent_role_id, role
            from     cmap_parent_role
            order by role
        ],
        { Columns => {} }
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'parent_roles_view'}, 
        { parent_roles => $parent_roles }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub physical_map_view {
    my ( $apr, $db, $t ) = @_;
    my $physical_map_id  = $apr->param('physical_map_id') or die 'No map id';
    my $order_by         = $apr->param('order_by') || 'position_start';

    my $sth = $db->prepare(
        q[
            select map.physical_map_id, map.map_name,
                   map.position_start, map.position_stop,
                   ms.map_study_id, ms.map_study_name,
                   mt.map_type
            from   cmap_physical_map map, cmap_map_study ms, cmap_map_type mt
            where  map.physical_map_id=?
            and    map.map_study_id=ms.map_study_id
            and    ms.map_type_id=mt.map_type_id
        ]
    );
    $sth->execute( $physical_map_id );
    my $map = $sth->fetchrow_hashref;

    $map->{'features'} = $db->selectall_arrayref(
        qq[
            select   mp.physical_map_position_id, mp.map_position_name,
                     mp.position_start, mp.position_stop,
                     f.feature_id, f.feature_name, f.feature_type_id,
                     ft.feature_type
            from     cmap_physical_map_position mp, cmap_feature f, 
                     cmap_feature_type ft
            where    mp.physical_map_id=?
            and      mp.feature_id=f.feature_id
            and      f.feature_type_id=ft.feature_type_id
            order by $order_by
        ],
        { Columns => {} }, ( $physical_map_id )
    );

    $map->{'clones'} = $db->selectall_arrayref(
        qq[
            select   f.feature_id, f.feature_name as clone_name,
                     mp.position_start, mp.position_stop
            from     cmap_physical_map_position mp, 
                     cmap_feature f,
                     cmap_feature_type ft
            where    mp.physical_map_id=?
            and      mp.feature_id=f.feature_id
            and      f.feature_type_id=ft.feature_type_id
            and      upper(ft.feature_type)=?
            order by position_start, clone_name
        ],
        { Columns => {} }, ( $physical_map_id, CLONE )
    );

    my $output; 
    $t->process( TEMPLATE->{'physical_map_view'}, { map => $map, dump=>Dumper($map) }, \$output ) 
        or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub species_create {
    my ( $apr, $db, $t ) = @_;

    my $output; 
    $t->process( 
        TEMPLATE->{'species_create'},
        {}, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub species_edit {
    my ( $apr, $db, $t ) = @_;

    my $sth = $db->prepare(
        q[
            select   species_id, common_name, full_name
            from     cmap_species
            where    species_id=?
        ]
    );
    $sth->execute( $apr->param('species_id') );
    my $species = $sth->fetchrow_hashref;

    my $output; 
    $t->process( 
        TEMPLATE->{'species_edit'},
        { 
            species => $species,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub species_insert {
    my ( $apr, $db, $t ) = @_;
    my $common_name = $apr->param('common_name') or die 'No common name';
    my $full_name   = $apr->param('full_name')   or die 'No full name';
    my $species_id  = next_number(
        db          => $db, 
        table_name  => 'cmap_species',
        id_field    => 'species_id',
    ) or die 'No species id';

    $db->do(
        q[ 
            insert
            into   cmap_species ( species_id, full_name, common_name )
            values ( ?, ?, ? )
        ],
        {}, ( $species_id, $full_name, $common_name )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=species_view' ); 
}

#-----------------------------------------------------
sub species_update {
    my ( $apr, $db, $t ) = @_;
    my $species_id  = $apr->param('species_id') or die 'No map type id';
    my $common_name = $apr->param('common_name') or die 'No common name';
    my $full_name   = $apr->param('full_name')   or die 'No full name';

    $db->do(
        q[ 
            update cmap_species
            set    common_name=?, full_name=?
            where  species_id=?
        ],
        {}, ( $common_name, $full_name, $species_id )
    );

    return redirect_home( $apr, ADMIN_HOME_URI.'?action=species_view' ); 
}

#-----------------------------------------------------
sub species_view {
    my ( $apr, $db, $t ) = @_;
    my $order_by = $apr->param('order_by') || 'common_name';

    my $species = $db->selectall_arrayref(
        qq[
            select   species_id, full_name, common_name
            from     cmap_species
            order by $order_by
        ],
        { Columns => {} }
    );

    my $output; 
    $t->process( 
        TEMPLATE->{'species_view'},
        { 
            species => $species,
        }, 
        \$output 
    ) or $output = $t->error;

    $apr->content_type('text/html');
    $apr->send_http_header;
    $apr->print( $output );
    return OK;
}

#-----------------------------------------------------
sub update_physical_features {
    my ( $apr, $db, $t ) = @_;
    
    $SIG{'CHLD'} = 'IGNORE';
    $ENV{'PATH'} = '/bin:/usr/bin';
    delete @ENV{'IFS', 'CDPATH', 'ENV', 'BASH_ENV'};
    defined ( my $kid = fork ) or die "Cannot fork: $!";
    my $path = $apr->dir_config('CMAP_BIN_DIR') || BIN_DIR;

    if ( $kid ) {
        my $output; 
        $t->process( 
            TEMPLATE->{'update_physical_features'},
            { 
                report => [ 
                    'This is a long-running process.',
                    "The external program's PID is $kid.",
                ],
            }, 
            \$output 
        ) or $output = $t->error;

        $apr->content_type('text/html');
        $apr->send_http_header;
        $apr->print( $output );
        return OK;
    } 
    else {
        $apr->cleanup_for_exec(); # untie the socket
        chdir '/'                or die "Can't chdir to /: $!";
        open STDIN, '/dev/null'  or die "Can't read /dev/null: $!";
        open STDOUT, '>/dev/null'
            or die "Can't write to /dev/null: $!";
        open STDERR, '>&STDOUT'  or die "Can't dup stdout: $!";
        setsid or die "Can't start a new session: $!";

        my $program = $path.'/fill_physical_marker_table.pl';
        exec or die "Cannot execute $program: $!";
    }    
}

1;

#-----------------------------------------------------
# All wholsome food is caught without a net or a trap.
# William Blake
#-----------------------------------------------------

=head1 NAME

Admin - curate comparative map data

=head1 SYNOPSIS

In httpd.conf:

  <Location /maps/admin>
    AuthType     Basic
    AuthName     "Maps Curation"
    AuthUserFile /usr/local/apache/passwd/passwords
    Require      valid-user
    SetHandler   perl-script
    PerlHandler  CSHL::ComparativeMaps::Admin
  </Location>

=head1 DESCRIPTION

This module is intended to provide a basic, web-based frontend for the
curation of the data for comparative maps.  As this time, it's fairly
limited to allowing the creation of new map studies, editing/deleting of
existing studies, and importing of data.  However, there are a couple
of scripts that must be run whenever new maps are imported (or
corrected) -- namely one that updates feature correspondences and one
that updates the precomputed "physical_feature" table.  Currently,
these must be run by hand.

It is strongly recommended that this <Location> include at least basic
authentication.  This will require you to read up on the "htpasswd"
program.  Essentially, you should be able to run:

  # htpasswd -c /path/to/passwd/file

This will "create" (-c) the file given as the last argument, so don't
use this if the file already exists.  You will be prompted for a user
name and password to save into that file.  After you've created this
file and edited your server's configuration file, restart Apache.

=head1 AUTHOR

Ken Y. Clark, kclark@logsoft.com

=head1 SEE ALSO

perl(1), htpasswd.

=cut
