package CSHL::AppConfig; 

#-----------------------------------------------------
# $Id: AppConfig.pm,v 1.2 2002/03/28 20:29:12 kclark Exp $
#
# File       : CSHL/Config.pm
# Programmer : Ken Y. Clark, kclark@cshl.org
# Created    : 2002/03/25
# Purpose    : object for retrieving local config options
#-----------------------------------------------------

=pod

=head1 NAME

CSHL::AppConfig - object for retrieving local config options

=head1 SYNOPSIS

For simple scalar retrievals:

  use CSHL::AppConfig;
  my $config = CSHL::AppConfig->new; 
  my $foo    = $config->get('foo');

For more complicated data structures:

  use CSHL::AppConfig qw[ :argcount ];
  my $config = CSHL::AppConfig->new; 
  $config->define('db_options', { ARGCOUNT => ARGCOUNT_HASH });
  my %db_options = %{ $config->get('db_options') };

=head1 DESCRIPTION

CSHL::AppConfig rovides a consistent object for retrieving local
configuration information.  It subclasses Andy Wardley's AppConfig
module in order to hide the location of a default config file and the
default arguments to AppConfig::new.  This is intended make it very
easy to use the module -- whether it actually is easy to use remains
to be seen.

If you're only getting simple scalar variables (those with an
"ARGCOUNT" of "ARGCOUNT_ONE"), then you're pretty much "all set."
Just create a new CSHL::AppConfig object and "get" the variable you
want.  If you need something that is a list or a hash, then you're
going to need to import the ":argcount" tagset and "define" your
variable with the appropriate "ARGCOUNT."  Please see the perldoc for
AppConfig and AppConfig::State for more on this.

NOTE:  Please "define" your variables before any call to "get,"
preferable just after creating the config object.

=head1 METHODS

=cut

use strict;
use vars qw[ $VERSION $Has_parsed_file @EXPORT @EXPORT_OK %EXPORT_TAGS ];
$VERSION = (qw$Revision: 1.2 $)[-1];

use AppConfig qw[ :argcount ];
use base qw[ AppConfig ];

#
# Some typeglob aliases to be able to 'use CSHL::AppConfig qw[:argcount]'
#
*EXPORT      = *AppConfig::EXPORT{ARRAY};
*EXPORT_OK   = *AppConfig::EXPORT_OK{ARRAY};
*EXPORT_TAGS = *AppConfig::EXPORT_TAGS{HASH};

#
# The location of the configuation file.
#
use constant CONFIG_FILE  => '/usr/local/apache/conf/gramene.conf';

#
# The default arguments to AppConfig->new.
#
use constant DEFAULT_ARGS => {
    CASE     => 0,               # case-insensitive
    CREATE   => 1,               # create undefined variables
    GLOBAL   => { 
        ARGCOUNT => ARGCOUNT_ONE # assume simple scalar values
    },
};

#-----------------------------------------------------
sub new {
=pod

=head2 new

  my $config = CSHL::AppConfig->new([ hashref ]);

Creates a new CSHL::AppConfig object.  To override the default
arguments to AppConfig::new, provide a hash reference as the only
argument to new.  For the acceptable arguments, perldoc
AppConfig::State.

By default, the config module will:

=over 4

=item * 

Be case-insensitive [i.e., $config->get('foo') and $config->get('FOO') will
return the same thing]

=item *

Create any variables you haven't explicity "defined"'d

=item *

Set new variables to have an ARGCOUNT of one (1) (i.e, they will be simple
scalars)

=back

=cut

    my $class  = shift;
    my $args   = shift || undef;
    my $config = AppConfig->new( $args || DEFAULT_ARGS );
    return bless $config, $class;
}

#-----------------------------------------------------
sub file { 
=pod

=head2 file

  $config->file('/path/to/my/groovy/conf.file');

Parses a configuration file.  Unless you want to parse a file other
than the default, you will never need to call this directly as any
call to "get" will make sure that we've parsed the configuration file
first.

=cut

    my $self = shift;
    my $file = shift || '';

    if ( $file || ! $Has_parsed_file ) {
        $self->SUPER::file( $file || CONFIG_FILE );
        $Has_parsed_file = 1;
    }
}

#-----------------------------------------------------
sub get {
=pod

=head2 get

  my $foo = $config->get('foo');

This method calls the "file" method before calling AppConfig's own
"get."

=cut

    my $self = shift;
    $self->file;
    $self->SUPER::get( @_ );
}

1;

#-----------------------------------------------------
# Where is the wisdom we have lost in knowledge?
# Where is the knowledge we have lost in information?
# T. S. Eliot
#-----------------------------------------------------

=pod

=head1 AUTHOR

Ken Y. Clark, kclark@cshl.org

=head1 SEE ALSO

perl(1), AppConfig.

=cut
