#!/usr/local/bin/perl

# file: drawRiceChromosome.pl

# description: It connects to the rice genome database of gramene, fetches the
#              rice genome information, and then draws the chromosomes 
#              according to the rice sequencing information.

# author: Xiaokang Pan

# date: 12/20/01


use strict;
use lib '/usr/local/gramene/lib/perl';
use DBI;
use GD;
use CSHL::Config;

my (%sequenced, %chromosome);

fetchData();  
drawChromosomes();


sub fetchData() {

  my $dbh = DBI->connect("dbi:Oracle:",
			 SequenceDBUser,
			 SequenceDBPassword);

  my $query1 = "select length, chromosomeid ".
               "from contig";
  my $sth1 = $dbh->prepare($query1);
  $sth1->execute();

  my $query2 = "select length, chromosome_id ".
               "from chromosome";
  my $sth2 = $dbh->prepare($query2);
  $sth2->execute();

  my ($dna_len, $chr_len, $chr_id, $chrid);
  while (($dna_len, $chr_id) = $sth1->fetchrow) {

      $sequenced{$chr_id} += $dna_len;   
  }

  while (($chr_len, $chrid) = $sth2->fetchrow) {

      $chromosome{$chrid} = $chr_len;   
  }

  $sth1->finish;
  $sth2->finish;
  $dbh->disconnect;
}


sub drawChromosomes() {

  my $im;
  my ($white, $black, $red, $blue, $yellow, $green);

  # create a new image
  my $im = new GD::Image(310,120);

  # allocate some colors
  $white = $im->colorAllocate(255,255,255);
  $black = $im->colorAllocate(0,0,0);
  $red = $im->colorAllocate(255,0,0);
  $blue = $im->colorAllocate(0,0,255);
  $yellow=$im->colorAllocate(225,225,0);
  $green=$im->colorAllocate(0,225,0);

  # make the background transparent and interlaced
  $im->transparent($white);
  $im->interlaced('true');
  print STDERR "Transparent=",$im->transparent()," white=$white\n";

  my $i;
  for ($i=1; $i<=12; $i++) {

    # Draw a blue oval
    my ($Cx,$Cy,$Lx,$Rx,$finished,$length);
    my $poly;
    $Cx = 12*2*$i;
    $length = $chromosome{$i}*100/53000000;
    $Cy = int(100-$length); 
    $Lx = int($Cx-5);
    $Rx = int($Cx+5);

    $poly = new GD::Polygon;

    $poly->addPt($Lx, 95);

    $poly->addPt($Lx+2, 98);

    $poly->addPt($Lx+4, 100);

    $poly->addPt($Lx+6, 100);

    $poly->addPt($Rx-2, 98);

    $poly->addPt($Rx, 95);

    $poly->addPt($Rx, $Cy+5);

    $poly->addPt($Rx-2, $Cy+2);

    $poly->addPt($Rx-4, $Cy);
   
    $poly->addPt($Lx+4, $Cy);

    $poly->addPt($Lx+2, $Cy+2);

    $poly->addPt($Lx, $Cy+5);
  
    $im->polygon($poly, $blue);

    $Lx = $Cx-5;
    $Rx = $Cx+5;
    $finished = $sequenced{$i}*100/53000000;
    if ($finished > 100) {
        $finished = 100;
    }
    
    if ((($length-$finished) > 0 && ($length-$finished) <= 2) || 
	(($length-$finished) >= 98 && ($length-$finished) < 100)) {

      $im->line($Lx+2,100-$finished,$Rx-2,100-$finished,$blue);

    } elsif ((($length-$finished) > 2 && ($length-$finished) < 5) ||
	     (($length-$finished) > 95 && ($length-$finished) < 98)) {

      $im->line($Lx+1,100-$finished,$Rx-1,100-$finished,$blue);

    } elsif (($length-$finished) <= 95 && ($length-$finished) >= 5) {

      $im->line($Lx+1,100-$finished,$Rx-1,100-$finished,$blue);
    }

    # fill it with green for sequenced portion of the genome
    $im->fill($Cx,98,$green);


    # fill it with yellow for left over 
    if (($length-$finished) >= 1) {
      $im->fill($Cx,100-$length+1,$yellow);
    }

    # mark it with number
    if ($i < 10) {
      $im->string(gdSmallFont,$Cx-2,105,$i,$black);
    } else {
      $im->string(gdSmallFont,$Cx-5,105,$i,$black);
    }
  }

  print STDERR "Transparent=",$im->transparent()," white=$white\n";

  # Convert the image to png and print it on standard output
  open (OUT, ">/usr/local/gramene/html/chr_image/chrnew.png") || die;

  # make sure we are writing to a binary stream
  binmode OUT;

  print OUT $im->png;
 
}

