#!/usr/bin/perl

#-----------------------------------------------------
# $Id: map_importer.pl,v 1.12 2002/04/18 18:07:41 kclark Exp $
#
# File       : map_importer.pl
# Programmer : Ken Y. Clark, kclark@cshl.org
# Created    : 2002/03/18
# Purpose    : import map study data into comparative maps database
#-----------------------------------------------------

use strict;
use DBI;
use IO::File;
use Pod::Usage;
use Getopt::Long;
use Data::Dumper;
use Term::ReadLine;

use CSHL::Config;
use CSHL::ComparativeMaps::Constants;
#use CSHL::ComparativeMaps::Admin::BaseObject();
use CSHL::ComparativeMaps::Admin::DataImport();
use CSHL::ComparativeMaps::Admin::UpdatePhysicalFeatures();

use vars qw[ $VERSION $TERM $BE_QUIET ];
$VERSION = (qw$Revision: 1.12 $)[-1];
$TERM    = Term::ReadLine->new;

#
# Turn off output buffering.
#
$| = 1;

#
# Get command-line options
#
my $show_help;    
my $show_version; 

GetOptions(
    'h|help'    => \$show_help,    # Show help and exit
    'q|quiet'   => \$BE_QUIET,     # Don't show status messages
    'v|version' => \$show_version, # Show version and exit
) or pod2usage(2);

pod2usage(0) if $show_help;
if ( $show_version ) {
    print "$0 Version: $VERSION\n";
    exit(0);
}

#
# Get file argument
#
my $file = shift || '';

my %dispatch =  (
    default                  => \&show_greeting,
    create                   => \&create_map_study,
    import                   => \&import_data,
    update_physical_features => \&update_physical_features,
    quit                     => sub { print "Namaste.\n"; exit(0) },
);

my $db = DBI->connect(MapDataSource, MapDBUser, MapDBPassword, MapDBOptions);

while ( 1 ) { $dispatch{ show_greeting() }->( $db, $file ) }

#-----------------------------------------------------
sub show_greeting {
    my $separator = '-=' x 10;
    my $action  =  show_menu(
        title   => join("\n", $separator, '  --= Main Menu =--  ', $separator),
        prompt  => 'What would you like to do?',
        display => 'display',
        return  => 'action',
        data    => [
            { 
                action  => 'create', 
                display => 'Create new map study' 
            },
            { 
                action  => 'import', 
                display => 'Import data for existing study' 
            },
            { 
                action  => 'update_physical_features', 
                display => 'Update Physical Features' 
            },
            { 
                action  => 'quit',   
                display => 'Quit' 
            },
        ],
    );
    return $action;
}

#-----------------------------------------------------
sub create_map_study {
    print "Creating new map study.\n";
    
    my ( $map_type_id, $map_type ) = show_menu(
        title   => 'Available Map Types',
        prompt  => 'What type of map?',
        display => 'map_type',
        return  => 'map_type_id,map_type',
        data     => $db->selectall_arrayref(
            q[
                select   mt.map_type_id, mt.map_type
                from     cmap_map_type mt
                order by map_type
            ],
            { Columns => {} },
        ),
    );
    die "No map types to select from.\n" unless $map_type_id;

    my ( $species_id, $common_name ) = show_menu(
        title   => 'Available Species',
        prompt  => 'What species?',
        display => 'common_name,full_name',
        return  => 'species_id,common_name',
        data     => $db->selectall_arrayref(
            q[
                select   s.species_id, s.common_name, s.full_name
                from     cmap_species s
                order by common_name
            ],
            { Columns => {} },
        ),
    );
    die "No species to select from.\n" unless $species_id;

    print "Map Study Name (long): ";
    chomp( my $map_study_name = <STDIN> || 'New map study' ); 

    print "Short Name [$map_study_name]: ";
    chomp( my $short_name = <STDIN> );
    $short_name ||= $map_study_name; 

    my $map_study_id = next_number(
        db           => $db,
        table_name   => 'cmap_map_study',
        id_field     => 'map_study_id',
    ) or die 'No map study id';


    print "OK to create study '$map_study_name'?\n[Y/n] ";
    chomp( my $answer = <STDIN> );
    return if $answer =~ m/^[n]/;

    $db->do(
        q[
            insert
            into   cmap_map_study
                   ( map_study_id, map_study_name, short_name,
                     species_id, map_type_id
                   )
            values ( ?, ?, ?, ?, ? )
        ],
        {},
        (
            $map_study_id, $map_study_name, $short_name,
            $species_id, $map_type_id
        )
    );

    print "Map study $map_study_name created\n";
}

#-----------------------------------------------------
sub import_data {
#
# Gathers the info to import physical or genetic maps.
#
    my ( $db, $file ) = @_;

    #
    # Make sure we have a file to parse.
    #
    if ( $file ) {
        print "OK to use '$file'? [Y/n] ";
        chomp( my $answer = <STDIN> );
        $file = '' if $answer =~ m/^[Nn]/;
    }

    while ( ! -r $file ) {
        print "Unable to read '$file'.\n" if $file;
        $file =  $TERM->readline( 'Where is the file? [q to quit] ');
        $file =~ s/^\s*|\s*$//g;
        return if $file =~ m/^[Qq]/;

    }

    #
    # Open the file.  If it's good, remember it.
    #
    my $fh = IO::File->new( $file ) or die "Can't read $file: $!";
    $TERM->addhistory($file); 

    #
    # Get the map type.
    #
    my ( $map_type_id, $map_type ) = show_menu(
        title   => 'Available Map Types',
        prompt  => 'Please select a map type',
        display => 'map_type',
        return  => 'map_type_id,map_type',
        data     => $db->selectall_arrayref(
            q[
                select   mt.map_type_id, mt.map_type
                from     cmap_map_type mt
                order by map_type
            ],
            { Columns => {} },
        ),
    );
    do { print "No map types to select from.\n"; return } unless $map_type_id;

    #
    # Get the species.
    #
    my ( $species_id, $species ) = show_menu(
        title   => "Available Species (for $map_type)",
        prompt  => 'Please select a species',
        display => 'common_name',
        return  => 'species_id,common_name',
        data     => $db->selectall_arrayref(
            q[
                select   distinct s.species_id, s.common_name
                from     cmap_species s,
                         cmap_map_study ms
                where    ms.species_id=s.species_id
                and      ms.map_type_id=?
                order by common_name
            ],
            { Columns => {} },
            ( $map_type_id )
        ),
    );
    do { print "No species to select from.\n"; return } unless $species_id;
    
    #
    # Get the map study.
    #
    my ( $map_study_id, $map_study_name ) = show_menu(
        title   => "Available Map Studies (for $map_type, $species)",
        prompt  => 'Please select a map study',
        display => 'map_study_name',
        return  => 'map_study_id,map_study_name',
        data    => $db->selectall_arrayref(
            q[
                select   ms.map_study_id, ms.map_study_name
                from     cmap_map_study ms
                where    ms.map_type_id=?
                and      ms.species_id=?
                and      ms.map_type_id=?
                order by map_study_name
            ],
            { Columns => {} },
            ( $map_type_id, $species_id, $map_type_id )
        ),
    );
    do { print "There are no map studies for that map type!\n"; return }
         unless $map_study_id;

    #
    # Ask whether to overwrite or append the data.
    #
    my $overwrite = show_menu(
        title   => 'Overwrite/Append',
        prompt  => 'Do you wish to Overwrite or Append this data? ',
        display => 'display',
        return  => 'value',
        data    => [
            { value => 1, display => 'Overwrite' },
            { value => 0, display => 'Append'    },
        ],
    );

    my $overwrite_yes_no = $overwrite ? 'Yes' : 'No';

    print join("\n",
        'OK to import?',
        "  File      : $file",
        "  Species   : $species",
        "  Map Type  : $map_type",
        "  Map Study : $map_study_name",
        "  Overwrite : $overwrite_yes_no",
        "[Y/n] "
    );
    chomp( my $answer = <STDIN> );
    return if $answer =~ /^[Nn]/;

    print "Importing data...\n";

    my $importer = CSHL::ComparativeMaps::Admin::DataImport->new( db => $db );
    $importer->import(
        map_study_id => $map_study_id,
        fh           => $fh,
        map_type     => $map_type,
        overwrite    => $overwrite,
        be_quiet     => $BE_QUIET,
    ) or do { print "Error: ", $importer->error, "\n"; return; };
}

#-----------------------------------------------------
sub update_physical_features {
    my %args   = @_;

    print "You can update ALL the physical-to-genetic relationships\n",
        "or just those for one genetic map study, one physical map\n",
        "study, or those for a combination of one genetic and one\n",
        "physical map study.\n";

    my $map_study_sql = q[
        select   ms.map_study_id, ms.short_name,
                 s.common_name
        from     cmap_map_study ms,
                 cmap_map_type mt,
                 cmap_species s
        where    ms.map_type_id=mt.map_type_id
        and      upper(mt.map_type)=?
        and      ms.species_id=s.species_id
        order by common_name, short_name
    ];

    #
    # Get any one genetic map study id.
    #
    my ( $genetic_map_study_id, $genetic_map_study_name ) = show_menu(
        title      => 'Genetic Map Studies',
        prompt     => 'Choose a genetic map study ',
        display    => 'common_name,short_name',
        return     => 'map_study_id,short_name',
        allow_null => 1,
        data       => $db->selectall_arrayref(
            $map_study_sql, { Columns => {} }, ( GENETIC )
        ),
    );

    #
    # Get any one physical map study id.
    #
    my ( $physical_map_study_id, $physical_map_study_name ) = show_menu(
        title      => 'Physical Map Studies',
        prompt     => 'Choose a physical map study ',
        display    => 'common_name,short_name',
        return     => 'map_study_id,short_name',
        allow_null => 1,
        data       => $db->selectall_arrayref(
            $map_study_sql, { Columns => {} }, ( PHYSICAL )
        ),
    );

    my $map_study_names = $genetic_map_study_name || $physical_map_study_name
        ? join( ' and ', map { qq['$_'] || () } 
                $genetic_map_study_name, $physical_map_study_name
          )
        : 'ALL map studies'
    ;

    print "OK to update physical-to-genetic ",
          "relationships for $map_study_names?\n[Y/n] ";
    chomp( my $answer = <STDIN> );
    return if $answer =~ m/^[n]/;

    my $updater = 
        CSHL::ComparativeMaps::Admin::UpdatePhysicalFeatures->new( db => $db );

    $updater->update(
        dry_run               => 0,
        quiet                 => 0,
        no_prompt             => 0,
        genetic_map_study_id  => $genetic_map_study_id, 
        physical_map_study_id => $physical_map_study_id, 
    );
}

#-----------------------------------------------------
sub show_menu {
    my %args    = @_;
    my $data    = $args{'data'}   or return;
    my @return  = split(/,/, $args{'return'} ) 
                  or die "No return field(s) defined\n";
    my @display = split(/,/, $args{'display'});
    my $result;

    if ( scalar @$data > 1 ) {
        my $i      = 1;
        my %lookup = ();

        my $title = $args{'title'} || '';
        print $title ? "\n$title\n" : "\n";
        for my $row ( @$data ) {
            print "[$i] ", join(' : ', map { $row->{$_} } @display), "\n";
            $lookup{ $i } = scalar @return > 1 ? 
                [ map { $row->{$_} } @return ] : $row->{ $return[0] };
            $i++;
        }

        my $number;
        my $prompt = $args{'prompt'} || 'Please select';
        while ( 1 ) {
            print "\n$prompt", 
                $args{'allow_null'} ? '(0 for nothing)' : '',
                ': '
            ;
            chomp( $number = <STDIN> );
            if ( $args{'allow_null'} && $number == 0 ) {
                $result = undef;
                last;
            }
            elsif ( defined $lookup{ $number } ) {
                $result = $lookup{ $number }; 
                last;
            }
        }
    }
    elsif ( scalar @$data == 0 ) {
        $result = undef;
    }
    else {
        $result = [ map { $data->[0]->{ $_ } } @return ];
    }

    return wantarray ? defined $result ? @$result : undef : $result;
}

#-----------------------------------------------------
# Life is full of misery, loneliness, and suffering --
# and it's all over much too soon.
# Woody Allen
#-----------------------------------------------------

=pod

=head1 NAME

map_importer.pl - import map study data into comparative maps database

=head1 SYNOPSIS

  ./map_importer.pl [options] [map_study_data]

  Options:

    -h|help    Display help message
    -q|quiet   Don't print more than is necessary
    -v|version Display version

=head1 DESCRIPTION

This script is meant to be a command-line replacement for the
web-based creation and importing of map study data.  Why this giant
leap backwards?  These imports can take a *long* time, so it's awkward
to handle them in a web interface.

If the CSHL::* modules are not installed into your standard Perl
library path, be sure to have your PERL5LIB environment variable set
to their location (e.g., "/usr/local/apache/lib/perl") or to supply
that path to Perl when invoking the script, like so:

  perl -I/usr/local/apache/lib/perl map_importer.pl

=head1 AUTHOR

Ken Y. Clark, kclark@cshl.org

=head1 SEE ALSO

perl(1).

=cut
