#!/usr/bin/perl

#-----------------------------------------------------
# $Id: fill_physical_marker_table.pl,v 1.7 2002/03/27 22:21:33 kclark Exp $
#
# File       : fill_physical_map_table.pm
# Programmer : Ken Y. Clark, kclark@cshl.org
# Created    : 2002/03/18
# Purpose    : fill physical map table with relationships
#-----------------------------------------------------

use strict;
use Pod::Usage;
use Getopt::Long;
use DBI;
use Data::Dumper;

use CSHL::Config;
use CSHL::ComparativeMaps::Constants;

my $Version = (qw$Revision: 1.7 $)[-1];

#
# Get command-line options
#
my $show_help;    # Show help and exit.
my $show_version; # Show version and exit.
my $Quiet;        # Don't show status messages.
my $Dry_run;      # Don't execute any of the SQL.

GetOptions(
    'h|help'    => \$show_help,
    'q|quiet'   => \$Quiet,
    'v|version' => \$show_version,
    'd|dry'     => \$Dry_run,
) or pod2usage(2);

pod2usage(0) if $show_help;
if ( $show_version ) {
    print "$0 Version: $Version\n";
    exit(0);
}

#
# Open a database connection.
#
my $db = DBI->connect(MapDataSource, MapDBUser, MapDBPassword, MapDBOptions);

#
# Always start from scratch.
#
unless ( $Dry_run ) {
    my @tables = qw[ cmap_physical_feature cmap_physical_to_genetic_map ];
    print "OK to truncate tables:\n", map{ qq[  $_\n] } @tables;
    print '[Y/n] ';
    chomp( my $answer = <STDIN> );
    if ( $answer =~ m/^[Nn]/ ) {
        print "Not OK, exiting.  (Use -d or --dry for a dry run)\n\n";
        exit(0);
    }
    for my $table ( @tables  ) {
        Print("Truncating table $table.\n");
        $db->do( "truncate table $table" ) unless $Dry_run;
    }
}

#
# Find the physical map studies.
#
my @map_studies = @{ 
    $db->selectall_arrayref(
        q[
            select   ms.map_study_id, 
                     ms.short_name as map_study_name
            from     cmap_map_study ms, 
                     cmap_map_type mt
            where    ms.map_type_id=mt.map_type_id
            and      upper(mt.map_type)=?
            order by map_study_name
        ],
        { Columns => {} },
        ( PHYSICAL )
    )
};

Print("Processing ", scalar @map_studies, " physical map studies.\n");
my $separator = "\n#" . '-' x 50 . "\n";

for my $map_study ( @map_studies ) {
    #
    # Find all the contigs in this map study
    #
    my $map_study_id     = $map_study->{'map_study_id'};
    my $pmap_study_name  = $map_study->{'map_study_name'};
    my @physical_map_ids = @{ 
        $db->selectcol_arrayref(
            q[
                select   physical_map_id
                from     cmap_physical_map
                where    map_study_id=?
            ],
            {}, 
            ( $map_study_id )
        )
    };

    Print($separator, "Map study '$pmap_study_name' ($map_study_id) has ", 
        scalar @physical_map_ids, " contigs.", $separator);

    #
    # Use the embedded "Contig" package to create a new object for
    # each physical_map_id.
    #
    for my $physical_map_id ( @physical_map_ids ) {
        my $contig          = Contig->new( 
            physical_map_id => $physical_map_id,
            db              => $db,
        ) or next;

        my $contig_name     = $contig->name;
        my @genetic_maps    = $contig->genetic_maps;

        Print($separator, "$contig_name ($physical_map_id)",
            " has ", scalar @{ $contig->marker_ids }, " markers", 
            " & is related to ", scalar @genetic_maps,
            " genetic maps.\n\n");

        #
        # Compare the contig to each genetic map to which it is
        # related.
        #
        for my $genetic_map ( $contig->genetic_maps ) {
            my $genetic_map_id  = $genetic_map->{'genetic_map_id'};
            my $linkage_group   = $genetic_map->{'linkage_group'};
            my $gmap_study_name = $genetic_map->{'map_study_name'};

            Print("'$pmap_study_name-$contig_name' ($physical_map_id) vs. ",
                "'$gmap_study_name-$linkage_group' ($genetic_map_id)\n"
            );

            #
            # Even though $contig is an object, "compare" won't return
            # an object.  It returns a hashref with everything we need
            # to insert into the database.
            #
            my $built_contig = $contig->compare( $genetic_map );

            #
            # Insert each marker on the contig with the appropriate
            # info when this contig is held in relation to the current
            # genetic map.
            #
            for my $marker ( @{ $built_contig->{'markers'} } ) {
                my $marker_id = $marker->{'feature_id'};

                #
                # Markers on the contig can occur more than once on
                # the genetic map.
                #
                my @positions = @{ $marker->{'genetic_positions'} || [] }
                    ? @{ $marker->{'genetic_positions'} }
                    : ( undef )
                ;

                for my $genetic_position ( @positions ) {
                    my $position = $genetic_position->{'position'};

                    #
                    # Because we started by truncating this table, we
                    # shouldn't have to bother with updating a record,
                    # and I guess we'll assume that, if the record
                    # already exists, then it must be OK.
                    #
                    unless ( $Dry_run ) {
                        next if $db->selectrow_array(
                            q[
                                select count( feature_id )
                                from   cmap_physical_feature
                                where  feature_id=?
                                and    physical_map_id=?
                                and    genetic_map_id=?
                                and    genetic_position=?
                            ],
                            {}, 
                            (
                                $marker_id, 
                                $physical_map_id, 
                                $genetic_map_id,
                                $position,
                            )
                        );
                    }

                    $db->do(
                        q[
                            insert
                            into   cmap_physical_feature
                                   ( feature_id, 
                                     physical_map_id, 
                                     genetic_map_id,
                                     genetic_position, 
                                     is_discordant, 
                                     is_transposed 
                                   )
                            values ( ?, ?, ?, ?, ?, ? )
                        ],
                        {}, ( 
                            $marker_id, 
                            $physical_map_id, 
                            $genetic_map_id,
                            $position,
                            $marker->{'is_discordant'} || 0,
                            $marker->{'is_transposed'} || 0,
                        )
                    ) unless $Dry_run;

                    Print("Inserted marker ", 
                        $contig->marker_name( $marker_id ), " ($marker_id)",
                        " at position $position ",
                        "(", $genetic_position->{'map_position_name'}, ")",
                        " disc = '", $marker->{'is_discordant'}, "', ",
                        " trans = '", $marker->{'is_transposed'}, "'.\n"
                    );

                }
            }

            #
            # Now that we're done with the markers, insert the
            # contig's information (as a whole) into the database.
            #
            $db->do(
                q[ 
                    insert 
                    into   cmap_physical_to_genetic_map
                           ( physical_map_id, 
                             genetic_map_id,
                             genetic_position_start, 
                             genetic_position_stop,
                             no_concordant_features, 
                             is_discordant,
                             is_transposed
                           )
                    values ( ?, ?, ?, ?, ?, ?, ? )
                ],
                {},
                ( $physical_map_id, 
                  $genetic_map_id, 
                  $built_contig->{'start_cM'},
                  $built_contig->{'end_cM'}, 
                  $built_contig->{'no_concordant_features'} || 0,
                  $built_contig->{'is_discordant'}          || 0, 
                  $built_contig->{'is_transposed'}          || 0
                )
            ) unless $Dry_run;

            #
            # Let the user know what we've done, then move on.
            #
            Print(join("\n", map { $_ ? "  $_" : '' } 
                "Total # features = ".$built_contig->{'total_no'},
                "Start            = ".$built_contig->{'start_cM'},
                "End              = ".$built_contig->{'end_cM'},
                "# con. features  = ".$built_contig->{'no_concordant_features'},
                "Percent disc.    = ".$built_contig->{'percent_discordant'},
                "Is disc.         = ".$built_contig->{'is_discordant'} || 0,
                "Is trans.        = ".$built_contig->{'is_transposed'} || 0,
                '',
                '',
            ));
        }
    }
}

Print($separator, "Done\n");
exit(0);

#-----------------------------------------------------
sub Print {
#
# A simple way to suppress all the output if given "-q".  Send
# messages to STDERR so that it can be piped to a log of the session.
#
    return if $Quiet;
    warn @_;
}

#-----------------------------------------------------
package Contig;
#-----------------------------------------------------
use Data::Dumper;
use CSHL::ComparativeMaps::Constants;

#-----------------------------------------------------
sub new {
#
# Create a new contig, getting all the markers and 
# their genetic positions.
#
    my ( $class, %args ) = @_;
    my $physical_map_id = $args{'physical_map_id'} or die 'No physical map id';
    my $db              = $args{'db'}              or die 'No db connection';
    my $self            = { 
        physical_map_id => $physical_map_id,
        db              => $db, 
    };
    bless $self, $class;

    #
    # Get the contig's name and species id.
    #
    my ( $map_name, $species_id ) = $db->selectrow_array(
        q[
            select   pmap.map_name, ms.species_id
            from     cmap_physical_map pmap,
                     cmap_map_study ms
            where    pmap.physical_map_id=?
            and      pmap.map_study_id=ms.map_study_id
        ],
        { Columns => {} }, 
        ( $physical_map_id )
    );
    $self->{'map_name'}   = $map_name;
    $self->{'species_id'} = $species_id;

    #
    # Find all the markers that appear on this contig.  Put them in
    # their proper order on the contig.
    #
    my @markers = @{ 
        $db->selectall_arrayref(
            q[
                select   distinct pmp.feature_id,
                         pmp.position_start,
                         f.feature_name
                from     cmap_physical_map_position pmp,
                         cmap_feature f,
                         cmap_feature_type ft
                where    pmp.physical_map_id=?
                and      pmp.feature_id=f.feature_id
                and      f.feature_type_id=ft.feature_type_id
                and      upper(ft.feature_type)=?
                order by position_start
            ],
            { Columns => {} }, 
            ( $physical_map_id, MARKER )
        )
    } or return undef;
    my @marker_ids        = map { $_->{'feature_id'} } @markers;
    $self->{'marker_ids'} = \@marker_ids;

    #
    # Remember the names of the markers.
    #
    for my $marker ( @markers ) {
        my $feature_id   = $marker->{'feature_id'}   or next;
        my $feature_name = $marker->{'feature_name'} or next;
        $self->{'marker_names'}{ $feature_id } = $feature_name;
    }

    #
    # Find all the correspondences for those markers.  Use them
    # to create a lookup (e.g., A->B, B->A).
    #
    my @correspondences   = @{ 
        $db->selectall_arrayref(
            q[
                select feature_id1, feature_id2
                from   cmap_feature_correspondence
                where  feature_id1 in (].
                       join( ',', @marker_ids ).q[)
            ],
            { Columns => {} }
        )
    };

    for my $hr ( @correspondences ) {
        push @{ $self->{'correspondences'}{ $hr->{'feature_id1'} } }, 
            $hr->{'feature_id2'};
        push @{ $self->{'correspondences'}{ $hr->{'feature_id2'} } }, 
            $hr->{'feature_id1'};
    }

    #
    # Find all the genetic maps hit by the markers and aliases.
    #
    my @correspondence_ids  = map { $_->{'feature_id2'} } @correspondences;
    $self->{'genetic_maps'} = $db->selectall_arrayref(
        q[
            select   distinct gmp.genetic_map_id, 
                     gmap.linkage_group,
                     ms.short_name as map_study_name,
                     ms.species_id
            from     cmap_genetic_map_position gmp,
                     cmap_genetic_map gmap,
                     cmap_map_study ms
            where    gmp.feature_id in (].
                     join( ',', @marker_ids, @correspondence_ids ).q[)
            and      gmp.genetic_map_id=gmap.genetic_map_id
            and      gmap.map_study_id=ms.map_study_id
            order by map_study_name, linkage_group
        ],
        { Columns => {} }
    );

    #
    # Find all the possible map positions of this contig's markers on
    # all the genetic maps.
    #
    my @genetic_positions = @{
        $db->selectall_arrayref(
            q[
                select mp.feature_id,
                       mp.map_position_name,
                       mp.position_start as position, 
                       gmap.genetic_map_id,
                       gmap.linkage_group,
                       gmap.map_study_id,
                       ms.species_id
                from   cmap_genetic_map_position mp,
                       cmap_genetic_map gmap,
                       cmap_map_study ms
                where  mp.feature_id in (].
                       join( ',', @marker_ids, @correspondence_ids ).q[)
                and    mp.genetic_map_id=gmap.genetic_map_id
                and    gmap.map_study_id=ms.map_study_id
            ],
            { Columns => {} }
        )
    };

    #
    # Use the genetic positions to figure out the linkage group hit by
    # the most number of distinct map studies.
    #
    my %linkage_groups   = ();
    for my $locus ( @genetic_positions ) {
        my $linkage_group = $locus->{'linkage_group'};
        my $map_study_id  = $locus->{'map_study_id'};
        $linkage_groups{ $map_study_id }{ $linkage_group } = 1;
    }

    my %majority_linkage_groups;
    for my $map_study_id ( keys %linkage_groups ) {
        for my $lg ( keys %{ $linkage_groups{ $map_study_id } } ) {
            $majority_linkage_groups{ $lg }++;
        } 
    }
    my ( $majority_linkage_group, $last_number );
    while ( my ( $lg, $number ) = each %majority_linkage_groups ) {
        $majority_linkage_group = $lg if $number || 0 > $last_number;
        $last_number            = $number;
    }
    $self->{'majority_linkage_group'} = $majority_linkage_group;

    #
    # Sort out all the genetic positions by marker id so we can look
    # it up later.
    #
    for my $map_position ( @genetic_positions ) {
        my $feature_id      = $map_position->{'feature_id'};
        my $correspondences = $self->correspondences( $feature_id ) || [];

        #
        # We want to make sure that we're only recording the
        # genetic positions tagged according to the *contig's*
        # marker ids, not to a marker id that occurs on a genetic
        # map.  If we get a marker_id not on the contig, then it
        # must be one of the corresponding markers.
        #
        for my $id ( $feature_id, @$correspondences ) {
            if ( $self->marker_is_on_contig( $id ) ) {
                push @{ $self->{'genetic_positions'}{ $id } }, $map_position;
                last;
            }
        }
    } 

    return $self;
}

#-----------------------------------------------------
sub compare {
#
# This sub takes a genetic map.  It figures out the markers on the
# contig and which of those occur on the genetic map.  Then, it goes
# through the markers and figures which are discordant based on their
# species, linkage group, and how far they are from other markers on
# that linkage group.  It also tries to figure out which are
# transposed (occur in a different relative order on the contig than
# on the genetic map).  Once we have all that, then we can pronounce
# that the contig, as a whole, is discordant and/or transposed when
# shown in relation to a particular genetic map.
#
    my $self                    = shift;
    my $genetic_map             = shift or die 'No genetic map';
    my $db                      = $self->{'db'} or die 'No db connection';
    my $physical_map_id         = $self->physical_map_id or return;
    my $reference_map_id        = $genetic_map->{'genetic_map_id'};
    my $reference_species_id    = $genetic_map->{'species_id'};
    my $reference_linkage_group = $genetic_map->{'linkage_group'};

    #
    # Note each genetic position and the feature that occurs there.
    #
    my @marker_positions = ();
    for my $marker_id ( $self->marker_ids ) {
        my $found_markers     = 0;
        my @genetic_positions = 
            $self->genetic_positions( $marker_id, $reference_map_id );

        my $marker_name = $self->marker_name($marker_id);
#        warn "$marker_name no g positions = ", scalar @genetic_positions, "\n";
#        warn "GENETIC POS ($marker_name)\n", Dumper(@genetic_positions), "\n";

        #
        # Grab all the genetic positions for the linkage group of the
        # reference species.  
        #
        for my $map_position ( @genetic_positions ) {
            if ( 
                $map_position->{'species_id'}    == $reference_species_id &&
                $map_position->{'linkage_group'} == $reference_linkage_group
            ) {
                #
                # Set the "feature_id" to the one found on the contig
                # (it could be an alias of this feature).
                #
                $map_position->{'feature_id'} = $marker_id;
                push @marker_positions, $map_position; 
                $found_markers = 1;
            }
        }

        #
        # If the map position isn't for the same species and linkage
        # group, then we'll just record that the marker is there.
        #
        unless ( $found_markers ) {
            push @marker_positions, { feature_id => $marker_id };
        }
    }

#    warn "marker positions =\n", Dumper( @marker_positions ), "\n";

    #
    # Put the positions in ascending numerical order by their position
    # on the genetic (reference) map.  Remember that
    # $self->marker_ids() returns the marker_id's in the order they
    # occur on the contig.
    #
    @marker_positions = sort { $a->{'position'} <=> $b->{'position'} }
        @marker_positions;

    #
    # Go through the marker positions and record the genetic positions
    # by marker id if the position is on the reference map.  Also
    # figure out the first and last marker positions on the reference
    # map in order to know the span on the genetic map that markers
    # from the contig cover.
    #
    my ( %positions_by_marker_id, @defined_genetic_positions );
    for ( @marker_positions ) {
        my $position = $_->{'position'} or next;
        next unless $_->{'genetic_map_id'} == $reference_map_id;
        push @defined_genetic_positions, $position;

        push @{ $positions_by_marker_id{ $_->{'feature_id'} } }, $_;
    } 
    my $first_genetic_position = $defined_genetic_positions[  0 ]; 
    my $last_genetic_position  = $defined_genetic_positions[ -1 ];

    #
    # Go through the markers IN THEIR ORDER ON THE GENETIC MAP and
    # figure out which ones are discordant.
    #
    my %discordant_markers     = (); # Holds the IDs of discordant markers

    if ( $self->species_id == $reference_species_id ) {
        my %same_map_markers_seen = ();

        for my $i ( 0..$#marker_positions ) {
            my $marker         = $marker_positions[ $i ];
            my $feature_id     = $marker->{'feature_id'};
            my $cur_position   = $marker->{'position'};
            my $linkage_group  = $marker->{'linkage_group'};
            my $genetic_map_id = $marker->{'genetic_map_id'};

            #
            # A feature can occur on the correct linkage group for the
            # correct species more than once, but if it occurs on the
            # same genetic map as the one we're looking at, then we
            # don't want to consider any others.  So we'll skip
            # features that have already been seen on the reference
            # map.
            #
            next if $same_map_markers_seen{ $feature_id };

            #
            # If these are the same species and linkage group (but not
            # necessarily the same genetic map), then start checking
            # the distances between them.  Since we earlier tried to
            # sort these guys out by putting our current reference
            # map's position(s) first, then we should not consider
            # markers from other linkage groups if there's one on the
            # same map.
            #
            if ( 
                defined $cur_position &&
                $linkage_group == $reference_linkage_group
            ) { 
                my $prev_position = $i > 0           
                                    ? $marker_positions[$i-1]->{'position'}
                                    : undef;
                my $next_position = $i < $#marker_positions
                                    ? $marker_positions[$i+1]->{'position'}
                                    : undef;
                my $prev_interval = defined $prev_position 
                                    ? abs( $cur_position - $prev_position ) 
                                    : undef;
                my $next_interval = defined $next_position 
                                    ? abs( $cur_position - $next_position ) 
                                    : undef;

                #
                # Everytime a marker is too far away from it's
                # previous or next marker, its score is incremented.
                # Everytime its within tolerance, its score is
                # decremented.
                #
                my $score = 0;
                for my $interval ( $prev_interval, $next_interval ) {
                    if ( defined $interval ) {
                        $score += 
                            $interval < MAX_CM_DISTANCE_DISCORDANT ? 1 : -1;
                    }
                }

#                warn $self->marker_name($feature_id), ": ", 
#                     "cur = $cur_position, prev = $prev_position, ", 
#                     "next = $next_position, score = $score\n";

                #
                # A score of greater than zero signifies concordancy.
                # Anything greater than zero means discordancy.
                #
                $discordant_markers{ $feature_id } = ( $score >= 0 ) ? 0 : 1;

                #
                # Remember, markers can occur in more than one location on
                # the reference map, so we "push."
                #
                if ( $genetic_map_id == $reference_map_id ) {
#                    push @{ $positions_by_marker_id{ $feature_id } }, $marker;
                    $same_map_markers_seen{ $feature_id } = 1;
                }
            }
            #
            # If these are the same species but the marker has no
            # genetic position, then the feature on the contig must be
            # discordant.
            #
            else {
                $discordant_markers{ $feature_id } = 
                    $linkage_group == $reference_linkage_group ? 0 : 1;
            }

        }
    }

    #
    # Go through the markers IN THEIR ORDER ON THE CONTIG and
    # figure out which ones are transposed.
    #
    my $last_position          ; # The last cM position on the genetic map
    my %transposed_markers = (); # Holds the IDs of transposed markers
    my @finished_markers   = (); # Holds marker ID of processed markers
    my $i                  =  0; # The counter

    for my $marker_id ( $self->marker_ids ) {
        #
        # This will only give us positions on the reference map, which
        # is what we want.
        #
#        warn "Marker ", $self->marker_name( $marker_id ), " ($marker_id)\n";
        my @genetic_positions = 
            @{ $positions_by_marker_id{ $marker_id } || [] } or next;

        for my $position ( map { $_->{'position'} } @genetic_positions ) {
#        for my $position ( @genetic_positions ) {
#            warn "Position = '$position', last pos = '$last_position'\n";

            if ( $position < $last_position ) {
                my $previous_feature_id = $i > 0 
                    ? $finished_markers[ $i - 1 ] : undef;
                $transposed_markers{ $_ } = 1
                    for map { $_ || () } 
                    $marker_id, $previous_feature_id
                ;
            }

            $last_position = $position;

            push @finished_markers, $marker_id;
            $i++;
        }
    }

#    warn "xposed =\n", Dumper( %transposed_markers ), "\n";
#    warn "positions =\n", Dumper( %positions_by_marker_id ), "\n";

    #
    # Create a temporary array to hold each of the contig's marker's
    # info and it's attributes (discordant, transposed) after having
    # been compared to the genetic map.
    #
    my @markers = ();
    for my $marker_id ( $self->marker_ids ) {
        push @markers, {
            feature_id        => $marker_id,
            is_discordant     => $discordant_markers{ $marker_id } || 0,
            is_transposed     => $transposed_markers{ $marker_id } || 0,
            genetic_positions => $positions_by_marker_id{ $marker_id },
        }
    }

#    warn "markers =\n", Dumper( @markers ), "\n";

    #
    # Figure out if the contig as a whole is discordant.
    #
    my $total_no           = scalar @markers;
    my $no_bad             = grep { /1/ } values %discordant_markers;
    my $percent_concordant = sprintf( 
        "%d", ( ( $total_no - $no_bad ) / $total_no ) * 100 
    );
    my $no_anchored_markers   = grep { $_->{'genetic_positions'} } @markers;
    my $no_transposed_markers = grep { $_->{'is_transposed'}     } @markers;

    my $percent_discordant = 100 - $percent_concordant;

    #
    # Return a hashref with everything we need to insert the markers
    # and the contig in relation to the genetic map.
    #
    return {
        physical_map_id        => $physical_map_id,
        map_name               => $self->name,
        genetic_map_id         => $reference_map_id,
        linkage_group          => $genetic_map->{'linkage_group'},
        total_no               => $total_no,
        no_concordant_features => $total_no - $no_bad,
        percent_discordant     => $percent_discordant,
        is_discordant          => $percent_discordant > 50,
        is_transposed          => $no_anchored_markers==$no_transposed_markers,
        start_cM               => $first_genetic_position,
        end_cM                 => $last_genetic_position,
        markers                => \@markers,
    };
}

#-----------------------------------------------------
sub correspondences { 
#
# Returns an array ref of the feature ids corresponding to the arg.
#
    my $self      = shift;
    my $marker_id = shift or return;
    return $self->{'correspondences'}{ $marker_id };
}

#-----------------------------------------------------
sub genetic_maps {
#
# Returns an array[ref] of hashrefs containing the genetic maps that
# this contigs is related to.
#
    my $self = shift;
    return wantarray ? @{ $self->{'genetic_maps'} } : $self->{'genetic_maps'};
}

#-----------------------------------------------------
sub genetic_positions { 
#
# Returns an array[ref] of all the genetic positions (on all maps) of
# a given marker.
#
    my $self      = shift;
    my $marker_id = shift or return;
    my @positions = @{ $self->{'genetic_positions'}{ $marker_id } || [] };

#    warn "GENETIC POSTIONS: ", $self->marker_name( $marker_id ), "\n";
#    warn Dumper( @positions ), "\n";

    #
    # If passed the optional argument "reference_map_id," then we'll
    # try to sort out the positions for the marker putting the
    # position for the reference map first, all other later.
    #
    if ( my $reference_map_id = shift ) {
        my %position_lookup   = map { $_->{'genetic_map_id'}, $_ } @positions;
        my $reference_position;

        for my $genetic_map_id ( keys %position_lookup ) {
            if ( $genetic_map_id == $reference_map_id ) {
                $reference_position = $position_lookup{ $genetic_map_id };
                delete $position_lookup{ $genetic_map_id };
                last;
            }
        }

        @positions = map { $_ || () } ( 
            $reference_position, values %position_lookup 
        );
#        warn "SORTED\n", Dumper( @positions ), "\n";
    }

#    warn Dumper( @positions ), "\n";
    return wantarray ? @positions : \@positions;
}

#-----------------------------------------------------
sub name { 
#
# The contig's name
#
    shift()->{'map_name'} 
}

#-----------------------------------------------------
sub majority_linkage_group {
#
# The linkage group signified by the most number of map studies.
#
    return shift()->{'majority_linkage_group'}
}

#-----------------------------------------------------
sub marker_ids {
#
# Returns an array[ref] of all the feature ids of the markers on the
# contig.
#
    my $self = shift;
    return wantarray ? @{ $self->{'marker_ids'} } : $self->{'marker_ids'};
}

#-----------------------------------------------------
sub marker_is_on_contig {
#
# Given a marker id, will tell you if that is a marker that is on the
# contig.  This is useful if you want to know if you need to get a
# corresponding marker or can use the one you have.
#
    my $self     = shift;
    my $given_id = shift or return;
    for my $marker_id ( $self->marker_ids ) {
        return 1 if $marker_id == $given_id;
    }

    return;
}

#-----------------------------------------------------
sub marker_name {
#
# Returns the name of a marker when given the id.  This is used to
# turn the marker_id returned by $self->marker_ids() into a name, so
# it will only return the name of a marker which is on the contig (not
# the name of one of its aliases).
#
    my $self      = shift;
    my $marker_id = shift or return;
    return $self->{'marker_names'}{ $marker_id };
}

#-----------------------------------------------------
sub physical_map_id { 
#
# Returns the contig's database ID.
#
    shift()->{'physical_map_id'} 
}

#-----------------------------------------------------
sub species_id { 
#
# Returns the species ID of the contig.
#
    shift()->{'species_id'} 
}

#-----------------------------------------------------
# I have never yet met a man who was quite awake.
# How could I have looked him in the face?
# Henry David Thoreau
#-----------------------------------------------------

=pod

=head1 NAME

fill_physical_marker_table.pl - fill physical map table with relationships

=head1 SYNOPSIS

  ./fill_physical_marker_table.pl [options]

  Options:

    -h|help    Display help message
    -q|quiet   Don't print more than is necessary
    -v|version Display version

=head1 DESCRIPTION

This script will figure out the quality (discordant, transposed, etc.)
of a marker on a contig when shown in relation to any genetic map.

To capture a log of the session, redirect STDERR to a file, like so:

  ./fill_physical_marker_table.pl 2>log

If the CSHL::* modules are not installed into your standard Perl
library path, be sure to have your PERL5LIB environment variable set
to their location (e.g., "/usr/local/apache/lib/perl") or to supply
that path to Perl when invoking the script, like so:

  perl -I/usr/local/apache/lib/perl fill_physical_marker_table.pl

=head1 AUTHOR

Ken Y. Clark, kclark@cshl.org

=head1 SEE ALSO

perl(1).

=cut
